/* aan_idct.cpp:  The C version of the iDCT routine.
*********************************************************************************
*
*                    Copyright (c) 1998 Intel Corporation
*    
*    THIS SOURCE CODE IS PROVIDED "AS IS" WITH NO WARRANTIES WHATSOEVER, 
*    INCLUDING ANY WARRANTY OF MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR ANY 
*    PARTICULAR PURPOSE, OR ANY WARRANTY OTHERWISE ARISING OUT OF ANY PROPOSAL, 
*    SPECIFICATION OR SAMPLE. Intel disclaims all liability, including liability 
*    for infringement of any proprietary rights, relating to use of information 
*    in this specification. No license, express or implied, by estoppel or 
*    otherwise, to any intellectual property rights is granted herein, other 
*    than a royalty-free copyright license to use, copy, modify, distribute, and 
*    otherwise dispose of this source code and resulting object code in any 
*    format, for execution on Intel Architecture processors.
*
*    Microprocessors may contain design defects or errors known as errata which
*    may cause the product to deviate from published specifications. Current 
*    characterized errata for Intel microprocessors are available on request.
*
*
***********************************************************************************/	


#include "aan_idct.h"


// Shift data and check Bound then store it in 'out'.
#define SHIFT_AND_BOUND(tmp, in, out)\
    if ((tmp = in + 128) <= 0) {\
        out = 0;\
    } else if (tmp >= 255) {\
        out = 255;\
    } else {\
        out = tmp;\
    }


/*F*
////////////////////////////////////////////////////////////////////////////
// Name:		IDct_AAN
//
// Purpose:		Performs an inverse DCT on an 8x8 block.
//
// Context:		The AAN (Arai, Agui, and Nakajima) algorithm from
//				Trans. IEICE, vol. E 71(11), 1095-1097, Nov. 1988 is used.
//				The implementation was developed by Intel Corporation.
//
// Returns:		None.
//
// Parameters:
//				coef_block	Input:  A set of 1 DC and 63 AC coefficients.
//							Ouput:  An 8x8 raster of image values.
//
////////////////////////////////////////////////////////////////////////////
*F*/
void IDct_AAN(short *coef_block)
{
	int   r0, r1, r2, r3, r4, r5, r6, r7, tmp, i;
	short *lptr;
	short *out = coef_block;

	// Do the columns
	for( lptr=coef_block, i=8; i>0; i--, lptr++ )
	{
		r0 = lptr[40];
		r1 = lptr[24];
		r2  = r0 - r1;
		r3  = r0 + r1;

		r0 = lptr[8];
		r1 = lptr[56];
		r4  = r0 + r1;
		r0  = r0 - r1;

		r1 = SCALEM(b5*(r2-r0));
		r0 = SCALEM(b4*r0) - r1;

		r2 = r1 - SCALEM(b2*r2);

		r1 = r4 - r3;
		r4 = r4 + r3;

		r5 = r0 - r4;
		r6 = SCALEM(b3*r1) - r5;
		r7 = r6 + r2;

		r0 = lptr[0];
		r1 = lptr[32];
		r2 = r0 + r1;
		r3 = r0 - r1;

		r0 = lptr[16];
		r1 = lptr[48];
		tmp = r0 - r1;
		r0 = r0 + r1;

		r1 = SCALEM(b1*tmp) - r0;

		tmp = r2 + r0;
		r0 = r2 - r0;
		r2 = r3 + r1;
		r1 = r3 - r1;

		lptr[0]  = (tmp + r4);
		lptr[8]  = (r2 + r5);
		lptr[16] = (r1 + r6);
		lptr[24] = (r0 - r7);
		lptr[32] = (r0 + r7);
		lptr[40] = (r1 - r6);
		lptr[48] = (r2 - r5);
		lptr[56] = (tmp - r4);
	}

	// Do the rows
	for( lptr=coef_block, i=8; i>0; i--, lptr += 8 )
	{
		r0 = lptr[5];
		r1 = lptr[3];
		r2  = r0 - r1;
		r3  = r0 + r1;

		r0 = lptr[1];
		r1 = lptr[7];
		r4  = r0 + r1;
		r0  = r0 - r1;

		r1 = SCALEM(b5*(r2-r0));
		r0 = SCALEM(b4*r0) - r1;

		r2 = r1 - SCALEM(b2*r2);

		r1 = r4 - r3;
		r4 = r4 + r3;

		r5 = r0 - r4;
		r6 = SCALEM(b3*r1) - r5;
		r7 = r6 + r2;

		r0 = lptr[0];
		r1 = lptr[4];
		r2 = r0 + r1;
		r3 = r0 - r1;

		r0 = lptr[2];
		r1 = lptr[6];
		tmp = r0 - r1;
		r0 = r0 + r1;

		r1 = SCALEM(b1*tmp) - r0;

		tmp = r2 + r0;
		r0 = r2 - r0;
		r2 = r3 + r1;
		r1 = r3 - r1;

		// Remove the 5-EXTRA-fractional bits by using SCALEE.
		SHIFT_AND_BOUND(r3, SCALEE(tmp + r4), out[0]);
		SHIFT_AND_BOUND(r3, SCALEE(r2 + r5), out[1]);
		SHIFT_AND_BOUND(r3, SCALEE(r1 + r6), out[2]);
		SHIFT_AND_BOUND(r3, SCALEE(r0 - r7), out[3]);
		SHIFT_AND_BOUND(r3, SCALEE(r0 + r7), out[4]);
		SHIFT_AND_BOUND(r3, SCALEE(r1 - r6), out[5]);
		SHIFT_AND_BOUND(r3, SCALEE(r2 - r5), out[6]);
		SHIFT_AND_BOUND(r3, SCALEE(tmp - r4), out[7]);
		out += 8;
	}

} // end of IDct_AAN
